let offscreenCreated = false;
let animationTimer = null;

const ICON_IDLE = {
  16: 'icons/icon-idle-16.png',
  48: 'icons/icon-idle-48.png',
  128: 'icons/icon-idle-128.png'
};

// --- Animated icon rendering (simplified: 1 sine wave, solid red) ---

function drawFrame(size, t) {
  const canvas = new OffscreenCanvas(size, size);
  const ctx = canvas.getContext('2d');

  const barCount = 5;
  const pad = Math.round(size * 0.12);
  const gap = Math.max(1, Math.round(size * 0.06));
  const totalGap = (barCount - 1) * gap;
  const barW = (size - 2 * pad - totalGap) / barCount;
  const r = barW / 2;
  const maxH = size - 2 * pad;

  const base = [0.45, 0.75, 1.0, 0.75, 0.45];
  ctx.fillStyle = '#ef4444';

  for (let i = 0; i < barCount; i++) {
    const wave = Math.sin(t * 3 + i * 1.4);
    const scale = 0.55 + 0.45 * wave;
    const barH = Math.max(barW, base[i] * scale * maxH);
    const x = pad + i * (barW + gap);
    const y = (size - barH) / 2;

    ctx.beginPath();
    ctx.roundRect(x, y, barW, barH, r);
    ctx.fill();
  }

  return ctx.getImageData(0, 0, size, size);
}

function startAnimation() {
  if (animationTimer) return;
  const t0 = performance.now();
  animationTimer = setInterval(() => {
    const t = (performance.now() - t0) / 1000;
    const imageData = {
      16: drawFrame(16, t),
      32: drawFrame(32, t)
    };
    chrome.action.setIcon({ imageData });
  }, 70);
}

function stopAnimation() {
  if (animationTimer) {
    clearInterval(animationTimer);
    animationTimer = null;
  }
  chrome.action.setIcon({ path: ICON_IDLE });
}

// --- State ---

async function getState() {
  const data = await chrome.storage.session.get(['recording', 'startTime', 'bitrate', 'boost']);
  return {
    recording: data.recording || false,
    startTime: data.startTime || null,
    bitrate: data.bitrate || 192,
    boost: data.boost || 1
  };
}

async function setState(recording, startTime, bitrate, boost) {
  await chrome.storage.session.set({ recording, startTime, bitrate, boost });
  if (recording) {
    startAnimation();
  } else {
    stopAnimation();
  }
}

// Restore animation on service worker restart
getState().then(state => {
  if (state.recording) startAnimation();
});

// --- Offscreen document ---

async function ensureOffscreenDocument() {
  if (offscreenCreated) return;
  const existingContexts = await chrome.runtime.getContexts({
    contextTypes: ['OFFSCREEN_DOCUMENT']
  });
  if (existingContexts.length > 0) {
    offscreenCreated = true;
    return;
  }
  await chrome.offscreen.createDocument({
    url: 'offscreen.html',
    reasons: ['USER_MEDIA'],
    justification: 'Recording tab audio via MediaRecorder'
  });
  offscreenCreated = true;
}

// --- Message handling ---

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.action === 'getStatus') {
    getState().then(sendResponse);
    return true;
  }

  if (msg.action === 'startRecording') {
    getState().then(async (state) => {
      if (state.recording) {
        sendResponse({ success: false, error: 'Already recording.' });
        return;
      }
      try {
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tabs || tabs.length === 0) {
          sendResponse({ success: false, error: 'No active tab found.' });
          return;
        }
        const tab = tabs[0];
        if (tab.url && (tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://'))) {
          sendResponse({ success: false, error: 'Cannot capture Chrome internal pages.' });
          return;
        }
        const streamId = await chrome.tabCapture.getMediaStreamId({ targetTabId: tab.id });
        await ensureOffscreenDocument();
        const bitrate = msg.bitrate || 192;
        const boost = msg.boost || 1;
        chrome.runtime.sendMessage({ action: 'offscreen-start', streamId, bitrate, boost });
        const startTime = Date.now();
        await setState(true, startTime, bitrate, boost);
        sendResponse({ success: true, startTime });
      } catch (err) {
        const name = err.name || '';
        const message = (err.message || '').toLowerCase();
        let error;
        if (name === 'NotAllowedError' || message.includes('permission')) {
          error = 'Permission denied. Try reloading the tab.';
        } else if (message.includes('cannot capture') || message.includes('no stream')) {
          error = 'This tab cannot be captured.';
        } else {
          error = err.message || 'Failed to start recording.';
        }
        sendResponse({ success: false, error });
      }
    });
    return true;
  }

  if (msg.action === 'stopRecording') {
    getState().then(async (state) => {
      if (!state.recording) {
        sendResponse({ success: false, error: 'Not recording.' });
        return;
      }
      chrome.runtime.sendMessage({ action: 'offscreen-stop' });
      await setState(false, null, state.bitrate, state.boost);
      sendResponse({ success: true });
    });
    return true;
  }

  if (msg.action === 'offscreen-complete') {
    if (!msg.blobUrl) return false;
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);
    const filename = `simple-audio-capture-dot-com-${timestamp}.mp3`;
    chrome.downloads.download({
      url: msg.blobUrl,
      filename,
      saveAs: false
    });
    return false;
  }
});
