// Realtime MP3 encoding — encodes during recording so stop is instant.
// Uses ScriptProcessorNode to get raw PCM and feeds it to lamejs on the fly.
// AnalyserNode sends frequency data to popup for real audio visualization.

let audioCtx = null;
let processor = null;
let analyser = null;
let gainNode = null;
let stream = null;
let mp3encoder = null;
let mp3Data = [];
let levelInterval = null;
const levelChannel = new BroadcastChannel('audio-levels');

chrome.runtime.onMessage.addListener((msg) => {
  if (msg.action === 'offscreen-start') {
    startRecording(msg.streamId, msg.bitrate || 192, msg.boost || 1);
  }
  if (msg.action === 'offscreen-stop') {
    stopRecording();
  }
});

async function startRecording(streamId, bitrate, boost) {
  stream = await navigator.mediaDevices.getUserMedia({
    audio: {
      mandatory: {
        chromeMediaSource: 'tab',
        chromeMediaSourceId: streamId
      }
    }
  });

  audioCtx = new AudioContext();
  const source = audioCtx.createMediaStreamSource(stream);

  const sampleRate = audioCtx.sampleRate;
  const channels = 2;
  mp3encoder = new lamejs.Mp3Encoder(channels, sampleRate, bitrate);
  mp3Data = [];

  // GainNode for volume boost (applied to recording + playback)
  gainNode = audioCtx.createGain();
  gainNode.gain.value = boost;
  source.connect(gainNode);

  // Analyser for real-time frequency data sent to popup
  analyser = audioCtx.createAnalyser();
  analyser.fftSize = 64;
  analyser.smoothingTimeConstant = 0.75;
  gainNode.connect(analyser);

  levelInterval = setInterval(() => {
    if (!analyser) return;
    const data = new Uint8Array(analyser.frequencyBinCount);
    analyser.getByteFrequencyData(data);
    levelChannel.postMessage(Array.from(data));
  }, 80);

  // ScriptProcessor captures raw PCM and passes audio through to speakers
  processor = audioCtx.createScriptProcessor(4096, channels, channels);

  processor.onaudioprocess = (e) => {
    const inputL = e.inputBuffer.getChannelData(0);
    const inputR = e.inputBuffer.getChannelData(1);

    // Pass through to output so user can still hear audio
    e.outputBuffer.getChannelData(0).set(inputL);
    e.outputBuffer.getChannelData(1).set(inputR);

    // Encode to MP3 in realtime
    const buf = mp3encoder.encodeBuffer(
      floatTo16Bit(inputL),
      floatTo16Bit(inputR)
    );
    if (buf.length > 0) mp3Data.push(buf);
  };

  gainNode.connect(processor);
  processor.connect(audioCtx.destination);
}

function stopRecording() {
  if (levelInterval) {
    clearInterval(levelInterval);
    levelInterval = null;
  }
  if (processor) {
    processor.disconnect();
    processor = null;
  }
  analyser = null;
  if (gainNode) {
    gainNode.disconnect();
    gainNode = null;
  }
  if (stream) {
    stream.getTracks().forEach(t => t.stop());
    stream = null;
  }
  if (audioCtx) {
    audioCtx.close();
    audioCtx = null;
  }

  // Flush remaining MP3 frames
  if (mp3encoder) {
    const end = mp3encoder.flush();
    if (end.length > 0) mp3Data.push(end);
    mp3encoder = null;
  }

  if (mp3Data.length === 0) {
    chrome.runtime.sendMessage({ action: 'offscreen-complete', blobUrl: '' });
    return;
  }

  const blob = new Blob(mp3Data, { type: 'audio/mp3' });
  mp3Data = [];
  const blobUrl = URL.createObjectURL(blob);
  chrome.runtime.sendMessage({ action: 'offscreen-complete', blobUrl });
}

function floatTo16Bit(samples) {
  const int16 = new Int16Array(samples.length);
  for (let i = 0; i < samples.length; i++) {
    const s = Math.max(-1, Math.min(1, samples[i]));
    int16[i] = s < 0 ? s * 0x8000 : s * 0x7FFF;
  }
  return int16;
}
