const startBtn = document.getElementById('startBtn');
const stopBtn = document.getElementById('stopBtn');
const timerEl = document.getElementById('timer');
const statusEl = document.getElementById('status');
const settingsRow = document.getElementById('settingsRow');
const qualitySelect = document.getElementById('qualitySelect');
const boostSelect = document.getElementById('boostSelect');
const canvas = document.getElementById('waveform');
const ctx = canvas.getContext('2d');

let startTime = null;
let animFrame = null;
let isRecording = false;
let audioLevels = null;
let smoothLevels = new Float32Array(32);

// Hi-DPI canvas
function sizeCanvas() {
  const dpr = window.devicePixelRatio || 1;
  const rect = canvas.getBoundingClientRect();
  canvas.width = rect.width * dpr;
  canvas.height = rect.height * dpr;
  ctx.scale(dpr, dpr);
}
sizeCanvas();

const W = canvas.getBoundingClientRect().width;
const H = canvas.getBoundingClientRect().height;

// Listen for audio level data from offscreen document via BroadcastChannel
const levelChannel = new BroadcastChannel('audio-levels');
levelChannel.onmessage = (e) => {
  audioLevels = e.data;
};

// --- Waveform rendering ---

function drawWave(t) {
  ctx.clearRect(0, 0, W, H);
  const midY = H / 2;

  if (!isRecording) {
    // Idle: gentle breathing line
    ctx.beginPath();
    ctx.strokeStyle = 'rgba(156,163,175,0.2)';
    ctx.lineWidth = 1.5;
    for (let x = 0; x < W; x++) {
      const y = midY + 1.5 * Math.sin(x * 0.04 + t * 0.8);
      x === 0 ? ctx.moveTo(x, y) : ctx.lineTo(x, y);
    }
    ctx.stroke();
    return;
  }

  // Smooth incoming frequency data for visual stability
  const levels = audioLevels || new Array(32).fill(0);
  const alpha = 0.35;
  for (let i = 0; i < smoothLevels.length; i++) {
    const target = i < levels.length ? levels[i] : 0;
    smoothLevels[i] += (target - smoothLevels[i]) * alpha;
  }

  const maxAmp = H * 0.4;
  const binCount = smoothLevels.length;

  // Interpolate frequency bin amplitude at normalized x position
  function getAmp(nx) {
    const pos = nx * (binCount - 1);
    const i = Math.floor(pos);
    const f = pos - i;
    const v1 = smoothLevels[Math.min(i, binCount - 1)] / 255;
    const v2 = smoothLevels[Math.min(i + 1, binCount - 1)] / 255;
    return (v1 + (v2 - v1) * f) * Math.sin(nx * Math.PI) * maxAmp;
  }

  // Filled shape (top half, then mirrored bottom)
  ctx.beginPath();
  for (let x = 0; x < W; x++) {
    const nx = x / W;
    const amp = getAmp(nx);
    const wave = 0.7 + 0.3 * Math.sin(nx * 8 + t * 2);
    const y = midY - amp * wave;
    x === 0 ? ctx.moveTo(x, y) : ctx.lineTo(x, y);
  }
  for (let x = W - 1; x >= 0; x--) {
    const nx = x / W;
    const amp = getAmp(nx);
    const wave = 0.7 + 0.3 * Math.sin(nx * 8 + t * 2);
    const y = midY + amp * wave;
    ctx.lineTo(x, y);
  }
  ctx.closePath();
  const grad = ctx.createLinearGradient(0, midY - maxAmp, 0, midY + maxAmp);
  grad.addColorStop(0, 'rgba(239,68,68,0.05)');
  grad.addColorStop(0.5, 'rgba(239,68,68,0.25)');
  grad.addColorStop(1, 'rgba(239,68,68,0.05)');
  ctx.fillStyle = grad;
  ctx.fill();

  // Center stroke driven by audio
  ctx.beginPath();
  ctx.lineWidth = 1.5;
  for (let x = 0; x < W; x++) {
    const nx = x / W;
    const amp = getAmp(nx);
    const wave = Math.sin(nx * 10 + t * 2.5);
    const y = midY + amp * wave * 0.5;
    x === 0 ? ctx.moveTo(x, y) : ctx.lineTo(x, y);
  }
  const avgLevel = smoothLevels.reduce((a, b) => a + b, 0) / smoothLevels.length / 255;
  ctx.strokeStyle = `rgba(239,68,68,${0.5 + 0.5 * avgLevel})`;
  ctx.stroke();
}

// --- Animation loop (handles waveform + timer in one rAF) ---

function animate() {
  const t = performance.now() / 1000;
  drawWave(t);

  // Update timer every frame (no jitter vs setInterval)
  if (isRecording && startTime) {
    timerEl.textContent = formatTime(Date.now() - startTime);
  }

  animFrame = requestAnimationFrame(animate);
}
animate();

// --- Timer ---

function formatTime(ms) {
  const totalSec = Math.floor(ms / 1000);
  const min = String(Math.floor(totalSec / 60)).padStart(2, '0');
  const sec = String(totalSec % 60).padStart(2, '0');
  return `${min}:${sec}`;
}

// --- UI state ---

function setRecordingUI(recording, recordingStartTime) {
  isRecording = recording;
  if (recording) {
    startBtn.style.display = 'none';
    stopBtn.style.display = 'block';
    settingsRow.style.display = 'none';
    timerEl.classList.add('active');
    startTime = recordingStartTime || Date.now();
    statusEl.textContent = '';
    statusEl.classList.remove('success');
  } else {
    startBtn.style.display = 'block';
    stopBtn.style.display = 'none';
    settingsRow.style.display = 'flex';
    timerEl.classList.remove('active');
    timerEl.textContent = '00:00';
    audioLevels = null;
    smoothLevels.fill(0);
  }
}

function showSaved() {
  statusEl.textContent = 'Saved to Downloads';
  statusEl.classList.add('success');
  setTimeout(() => {
    statusEl.textContent = '';
    statusEl.classList.remove('success');
  }, 3000);
}

// On popup open, check if already recording and restore quality setting
chrome.runtime.sendMessage({ action: 'getStatus' }, (resp) => {
  if (resp && resp.recording) {
    setRecordingUI(true, resp.startTime);
  }
  if (resp && resp.bitrate) {
    qualitySelect.value = String(resp.bitrate);
  }
  if (resp && resp.boost) {
    boostSelect.value = String(resp.boost);
  }
});

startBtn.addEventListener('click', () => {
  startBtn.disabled = true;
  statusEl.textContent = '';
  statusEl.classList.remove('success');

  const bitrate = parseInt(qualitySelect.value, 10);
  const boost = parseFloat(boostSelect.value);
  chrome.runtime.sendMessage({ action: 'startRecording', bitrate, boost }, (resp) => {
    startBtn.disabled = false;
    if (resp && resp.success) {
      setRecordingUI(true, resp.startTime);
    } else {
      statusEl.textContent = resp?.error || 'Failed to start recording.';
    }
  });
});

stopBtn.addEventListener('click', () => {
  stopBtn.disabled = true;

  chrome.runtime.sendMessage({ action: 'stopRecording' }, (resp) => {
    stopBtn.disabled = false;
    if (resp && resp.success) {
      setRecordingUI(false);
      showSaved();
    } else {
      statusEl.textContent = resp?.error || 'Failed to save.';
    }
  });
});
